#ifndef SBCLIB_ARRAY_1D

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <rnd/Mini.c>
#include <rnd/Max.c>

#ifndef REAL
#define REAL double // Assume double precision
#endif

REAL *array_1d(const unsigned s)
{
	REAL *a=(REAL *)malloc((1+s)*sizeof(REAL));
	*(unsigned *)a=s; // Assumes REAL is at least as long as unsigned
	return a+1;
}

void array_1d_free(REAL *a)
{
	free(a-1);
}

#define array_1d_m(A) (*(unsigned *)((A)-1))

REAL *array_1d_zeroed(const unsigned s)
{
	REAL *a=(REAL *)malloc((1+s)*sizeof(REAL));
	*(unsigned *)a=s; // Assumes REAL is at least as long as unsigned
	for (unsigned n=s;n>0;n--) a[n]=0;
	return a+1;
}

#include <rnd/explode.c>

REAL *array_1d_fromCSV(const char *s)
{ // Constructor from a string like "1.0,0.5,0.33"
	char **e=explode(",",s);
	int n,m=explodesegs(e); REAL *ret=array_1d(m);
	for (n=m-1;n>=0;n--) ret[n]=atof(e[n]);
	explodefree(e);
	return ret;
}

void array_1d_zero(REAL *a)
{
	int n,m=array_1d_m(a);
	for (n=m-1;n>=0;n--) a[n]=0;
}

void array_1d_fill(REAL *a,const REAL x)
{
	int n,m=array_1d_m(a);
	for (n=m-1;n>=0;n--) a[n]=x;
}

REAL *array_1d_copy(const REAL *c)
{ // Constructor
	unsigned m=array_1d_m(c); REAL *a=(REAL *)malloc((1+m)*sizeof(REAL));
	memcpy(a,c-1,(1+m)*sizeof(REAL));
	return a+1;
}

void array_1d_eq(REAL *a,const REAL *b)
{
	memcpy(a,b,Mini(array_1d_m(a),array_1d_m(b))*sizeof(REAL));
}

void array_1d_extendby(REAL **pa,const unsigned x)
{
	unsigned m=array_1d_m(*pa); REAL *a=(REAL *)malloc((1+m+x)*sizeof(REAL));
	*(unsigned *)a=m+x;
	memcpy(a+1,*pa,m*sizeof(REAL));
	free(*pa-1); *pa=a+1;
}

void array_1d_extendzeroed(REAL **pa,const unsigned nm)
{
	unsigned m=array_1d_m(*pa);
	if (m>=nm) return;
	REAL *a=(REAL *)malloc((1+nm)*sizeof(REAL));
	*(unsigned *)a=nm;
	memcpy(a+1,*pa,m*sizeof(REAL));
	for (unsigned n=nm;n>m;n--) a[n]=0;
	free(*pa-1); *pa=a+1;
}

#if defined(__stdio_h__) || defined(_STDIO_H_) || defined(_STDIO_H) || defined(_INC_STDIO) || defined(_STDIO_DEFINED)
void array_1d_print(const REAL *a,const char *nfmt="%lg",const char *sep=" ",const char *begin="[ ",const char *end=" ]\n")
{
	unsigned n,m=array_1d_m(a);
	puts(begin);
	for (n=0;n<m;n++) {printf(nfmt,a[n]); if (n+1<m) puts(sep);}
	puts(end);
}

void array_1d_printwrite(const REAL *a,FILE *out,const char *nfmt="%lg",const char *sep="\t",const char *begin="",const char *end="\n")
{
	unsigned n,m=array_1d_m(a);
	fputs(begin,out);
	for (n=0;n<m;n++) {fprintf(out,nfmt,a[n]); if (n+1<m) fputs(sep,out);}
	fputs(end,out);
}

void array_1d_printfile(const REAL *a,const char *filename,const char *nfmt="%lg",const char *sep="\t",const char *begin="",const char *end="\n")
{
	FILE *out=fopen(filename,"at");
	array_1d_printwrite(a,out,nfmt,sep,begin,end);
	fclose(out);
}

void array_1d_save(const REAL *a,const char *filename)
{ // Faster binary format
	FILE *out=fopen(filename,"wb");
	unsigned m=array_1d_m(a);
	fwrite(&m,1,sizeof(unsigned),out);
	fwrite(a,m,sizeof(REAL),out);
	fclose(out);
}

REAL *array_1d_load(const char *filename)
{
	FILE *in=fopen(filename,"rb");
	unsigned m;
	fread(&m,1,sizeof(unsigned),in);
	REAL *ret=array_1d(m);
	fread(ret,m,sizeof(REAL),in);
	fclose(in);
	return ret;
}
#endif

void array_1d_scale(REAL *a,const REAL k)
{
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]*=k;
}

void array_1d_add(REAL *a,const REAL *b)
{ // Make sure b is at least as big as a
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]+=b[n];
}

void array_1d_sub(REAL *a,const REAL *b)
{ // Make sure b is at least as big as a
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]-=b[n];
}

void array_1d_addsc(REAL *a,const REAL k,const REAL *b)
{ // Make sure b is at least as big as a
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]+=k*b[n];
}

void array_1d_mul(REAL *a,const REAL *b)
{ // Make sure b is at least as big as a
	for (int n=array_1d_m(a)-1;n>=0;n--) a[n]*=b[n];
}

REAL array_1d_max(const REAL *a)
{
	REAL ret=a[0];
	for (int n=array_1d_m(a)-1;n>0;n--) if (a[n]>ret) ret=a[n];
	return ret;
}

REAL array_1d_sum(const REAL *a)
{
	REAL ret=0;
	for (int n=array_1d_m(a)-1;n>=0;n--) ret+=a[n];
	return ret;
}

REAL array_1d_mean(const REAL *a)
{
	return array_1d_sum(a)/array_1d_m(a);
}

REAL array_1d_supnorm(const REAL *a)
{
	REAL ret=fabs(a[0]);
	for (int n=array_1d_m(a)-1;n>0;n--) ret=Max(ret,fabs(a[n]));
	return ret;
}

REAL array_1d_sumsq(const REAL *a)
{
	REAL ret=0;
	for (int n=array_1d_m(a)-1;n>=0;n--) ret+=a[n]*a[n];
	return ret;
}

REAL array_1d_norm(const REAL *a)
{
	return sqrt(array_1d_sumsq(a));
}

REAL array_1d_rms(const REAL *a)
{
	return sqrt(array_1d_sumsq(a)/array_1d_m(a));
}

void array_1d_qsort(REAL *a)
{ // Adapted from Numerical Recipes in C
	#define array_1d_qsort_NSTACK 51 // NSTACK is the required auxiliary storage
	int ir=array_1d_m(a)-1;
	if (ir<=0) return;
	const int M=7; // Here M is the size of subarrays sorted by straight insertion.  int not unsigned because compared with int difference!
	int i,j,k,l=0,istack[array_1d_qsort_NSTACK]; // lvector(1,NSTACK); Yeah, LCC's "dynamic arrays" really don't work well - use macro size!
	int jstack=0;
	REAL x,temp;
	#define SWAP(I,J) {temp=a[I]; a[I]=a[J]; a[J]=temp;}
	while (1)
	{ //Insertion sort when subarray small enough.
		if (ir-l<M)
		{
			for (j=l+1;j<=ir;j++)
			{
				x=a[j];
				for (i=j-1;i>=l;i--)
				{
					if (a[i]<=x) break;
					a[i+1]=a[i];
				}
				a[i+1]=x;
			}
			if (jstack==0) break;
			ir=istack[jstack--];// Pop stack and begin a new round of partitioning.
			l=istack[jstack--];
		}
		else
		{
			k=(l+ir)>>1; //Choose median of left, center, and right elements as partitioning element a. Also rearrange so that a[l] <= a[l+1] <= a[ir].
			SWAP(k,l+1);
			if (a[l]>a[ir]) SWAP(l,ir);
			if (a[l+1]>a[ir]) SWAP(l+1,ir);
			if (a[l]>a[l+1]) SWAP(l,l+1);
			i=l+1; //Initialize pointers for partitioning.
			j=ir;
			x=a[l+1]; //Partitioning element.
			while (1)
			{ //Beginning of innermost loop.
				do i++; while (a[i]<x); //Scan up to find element > x.
				do j--; while (a[j]>x); //Scan down to find element < x.
				if (j<i) break; //Pointers crossed. Partitioning complete.
				SWAP(i,j); //Exchange elements.
			} //End of innermost loop.
			a[l+1]=a[j]; //Insert partitioning element.
			a[j]=x;
			jstack+=2;
			//Push pointers to larger subarray on stack, process smaller subarray immediately.
			if (jstack>=array_1d_qsort_NSTACK) return; // nrerror("NSTACK too small in sort.");
			if (ir-i+1>=j-l)
			{
				istack[jstack]=ir;
				istack[jstack-1]=i;
				ir=j-1;
			}
			else
			{
				istack[jstack]=j-1;
				istack[jstack-1]=l;
				l=i;
			}
		}
	}
	#undef SWAP
	#undef array_1d_qsort_NSTACK
}

REAL array_1d_linear(const REAL *a,const REAL x)
{ 
	if (x<=0) return a[0];
	int m=array_1d_m(a);
	if (x>=m-1) return a[m-1];
	int n=(int)x; REAL f=x-n;
	return (1.0-f)*a[n]+f*a[n+1];
}

#ifdef SBCLIB_GRAPHICS
void array_1d_plot(const int ox,const int oy,const int z,const REAL *a,unsigned colmap(REAL))
{ // Draws a column vector analogous with array_2d_plot
	#ifdef SBCLIB_GRAPHICS_OPENGL
	glBegin(GL_QUADS);
	#endif
	int n,m=array_1d_m(a);
	for (n=m-1;n>=0;n--)
		#ifdef SBCLIB_GRAPHICS_OPENGL
		rect_inner
		#else
		rect
		#endif
			(ox,oy+n*z,z,z,colmap(a[n]));
	#ifdef SBCLIB_GRAPHICS_OPENGL
	glEnd();
	#endif
}
#endif

#define SBCLIB_ARRAY_1D
#endif
