#ifndef SBCLIB_ARRAY_2D
#include <stdlib.h>
#include <string.h>

#ifndef REAL
#define REAL double // Assume double precision
#endif

REAL **array_2d(const unsigned xs,const unsigned ys)
{
	REAL **a=(REAL **)malloc((2+xs)*sizeof(REAL *));
	*(unsigned *)a=xs; *(unsigned *)(a+1)=ys; a+=2;
	for (int x=xs-1;x>=0;x--) a[x]=(REAL *)malloc(ys*sizeof(REAL)); // Hmm, you know I could do this with only a single malloc by putting everything in one block
	return a;
}

#define array_2d_xs(A) (*(unsigned *)((A)-2))
#define array_2d_ys(A) (*(unsigned *)((A)-1))

void array_2d_free(REAL **a)
{
	for (int x=array_2d_xs(a)-1;x>=0;x--) free(a[x]);
	free(a-2);
}

REAL **array_2d_zeroed(const unsigned xs,const unsigned ys)
{
	int x,y; REAL **a=(REAL **)malloc((2+xs)*sizeof(REAL *));
	*(unsigned *)a=xs; *(unsigned *)(a+1)=ys; a+=2;
	for (x=xs-1;x>=0;x--)
	{
		a[x]=(REAL *)malloc(ys*sizeof(REAL));
		for (y=ys-1;y>=0;y--) a[x][y]=0;
	}
	return a;
}

REAL **array_2d_norows(const unsigned xs,const unsigned ys)
{ // Attach your own rows from e.g. malloced arrays
	REAL **a=(REAL **)malloc((2+xs)*sizeof(REAL *));
	*(unsigned *)a=xs; *(unsigned *)(a+1)=ys; a+=2;
	return a;
}

void array_2d_zero(REAL **a)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]=0;
}

REAL **array_2d_copy(const REAL * const *c)
{
	unsigned xs=array_2d_xs(c),sz=array_2d_ys(c)*sizeof(REAL);
	REAL **a=(REAL **)malloc((2+xs)*sizeof(REAL *));
	memcpy(a,c-2,2*sizeof(REAL *)); a+=2;
	for (int x=xs-1;x>=0;x--)
	{
		a[x]=(REAL *)malloc(sz);
		memcpy(a[x],c[x],sz);
	}
	return a;
}

void array_2d_eq(REAL **a,const REAL * const *b)
{ // a and b should be the same size, this just copies the data
	unsigned sz=array_2d_ys(b)*sizeof(REAL);
	for (int x=array_2d_xs(b)-1;x>=0;x--) memcpy(a[x],b[x],sz);
}

void array_2d_scale(REAL **a,const REAL k)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]*=k;
}

void array_2d_add(REAL **a,const REAL * const *b)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]+=b[x][y];
}

void array_2d_sub(REAL **a,const REAL * const *b)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]-=b[x][y];
}

void array_2d_addsc(REAL **a,const REAL k,REAL * const * const b)
{
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]+=k*b[x][y];
}

void array_2d_mul(REAL **a,const REAL **b)
{ // b should be at least as big as a
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) a[x][y]*=b[x][y];
}

REAL **array_2d_transpose(const REAL * const *a)
{
	int x,y,xs,ys; REAL **ret=array_2d(xs=array_2d_ys(a),ys=array_2d_xs(a));
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) ret[x][y]=a[y][x];
	return ret;
}

#if defined(__stdio_h__) || defined(_STDIO_H_) || defined(_STDIO_H) || defined(_INC_STDIO) || defined(_STDIO_DEFINED)
void array_2d_print(const REAL * const *a,const char *fmt="\t%lg")
{
	unsigned x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=0;x<xs;x++)
	{
		for (y=0;y<ys;y++) printf(fmt,a[x][y]);
		printf("\n");
	}
	printf("\n");
}

void array_2d_printwrite(const REAL * const *a,FILE *out,const char transpose=0)
{
	unsigned x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	if (transpose) for (y=0;y<ys;y++) for (x=0;x<xs;x++)
		fprintf(out,"%lg%c",a[x][y],x+1==xs?'\n':'\t');
	else for (x=0;x<xs;x++) for (y=0;y<ys;y++)
		fprintf(out,"%lg%c",a[x][y],y+1==ys?'\n':'\t');
	fprintf(out,"\n");
}

void array_2d_printfile(const REAL * const *a,const char *filename,const char transpose=0,const char *mode="wt")
{ // transpose included because array[small][large] is RAM efficient but wrong way for Excel
	FILE *out=fopen(filename,mode);
	array_2d_printwrite(a,out,transpose);
	fclose(out);
}

void array_2d_write(const REAL * const *a,FILE *out)
{
	unsigned x,xs=array_2d_xs(a),ys=array_2d_ys(a);
	fwrite(&xs,sizeof(unsigned),1,out);
	fwrite(&ys,sizeof(unsigned),1,out);
	for (x=0;x<xs;x++) fwrite(a[x],sizeof(REAL),ys,out);
}

void array_2d_save(const REAL * const *a,const char *filename)
{
	FILE *out=fopen(filename,"wb");
	array_2d_write(a,out);
	fclose(out);
}

REAL **array_2d_read(FILE *in)
{
	unsigned x,xs,ys;
	fread(&xs,sizeof(unsigned),1,in);
	fread(&ys,sizeof(unsigned),1,in);
	REAL **ret=array_2d(xs,ys);
	for (x=0;x<xs;x++) fread(ret[x],sizeof(REAL),ys,in);
	return ret;
}

void array_2d_readinto(FILE *in,REAL **a)
{
	unsigned x,xs,ys;
	fread(&xs,sizeof(unsigned),1,in);
	fread(&ys,sizeof(unsigned),1,in);
	for (x=0;x<xs;x++) fread(a[x],sizeof(REAL),ys,in);
}

REAL **array_2d_load(const char *filename)
{
	FILE *in=fopen(filename,"rb");
	REAL **ret=array_2d_read(in);
	fclose(in);
	return ret;
}

#include <math.h>
void array_2d_printstructure(const REAL * const *a,const REAL eps=1e-12)
{
	unsigned x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=0;x<xs;x++)
	{
		for (y=0;y<ys;y++)
		{
			if (fabs(a[x][y])<eps) printf(".");
			else if (fabs(a[x][y])*eps>1) printf("*");
			else if (a[x][y]>0) printf("+");
			else if (a[x][y]<0) printf("-");
			else printf("#");
		}
		printf("\n");
	}
	printf("\n");
}
#endif
#ifdef SBCLIB_BITMAPFONTS
void array_2d_display(const int ox,const int oy,const int cs,const REAL * const *a,const unsigned c)
{
	char str[100]; int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=ys-1;x>=0;x--) for (y=xs-1;y>=0;y--)
	{
		sprintf(str,"%lg",a[y][x]);
		bfwrite(ox+x*BF_height*cs,oy+y*BF_height,str,c);
	}
}
#endif
#ifdef SBCLIB_GRAPHICS
void array_2d_plot(const int ox,const int oy,const int z,const REAL * const *a,unsigned colmap(REAL))
{
	#ifdef SBCLIB_GRAPHICS_OPENGL
	glBegin(GL_QUADS);
	#endif
	int x,y,xs=array_2d_xs(a),ys=array_2d_ys(a);
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--)
		#ifdef SBCLIB_GRAPHICS_OPENGL
		rect_inner
		#else
		rect
		#endif
			(ox+x*z,oy+y*z,z,z,colmap(a[x][y]));
	#ifdef SBCLIB_GRAPHICS_OPENGL
	glEnd();
	#endif
}
#endif

#include <rnd/Mini.c>
#include <rnd/nonegi.c>
#include <rnd/Max.c>

REAL array_2d_bilinear(const REAL * const *a,const REAL x,const REAL y)
{ // Not sure if this is the fastest - might be worth trying to optimise a bit some time
	int xs=array_2d_xs(a),ys=array_2d_ys(a),
		ix=Mini((int)Max(x,0),xs-1),iy=Mini((int)Max(y,0),ys-1),
		jx=Mini(ix+1,xs-1),jy=Mini(iy+1,ys-1);
	REAL fx=x-ix,fy=y-iy;
	return (1.0-fx)*((1.0-fy)*a[ix][iy]+fy*a[ix][jy])+fx*((1.0-fy)*a[jx][iy]+fy*a[jx][jy]);
}

#ifdef SBCLIB_CSV
REAL **array_2d_fromCSV(const char *filename)
{
	csv c=csvload(filename);
	int x,y,xs,ys;
	REAL **ret=array_2d(xs=csvm(c),ys=csvrowm(c[0]));
	for (x=xs-1;x>=0;x--) for (y=ys-1;y>=0;y--) ret[x][y]=atof(c[x][y]);
	csvfree(c);
	return ret;
}
#endif

#define SBCLIB_ARRAY_2D
#endif
