#ifndef SBCLIB_LSET
// LSet = "Literal Set" library for C [Stephen Brooks 2001-11]
// It's literal because the objects are stored in a contiguous block of memory
#include <rnd/macros.c>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <rnd/Maxu.c>

typedef struct
{
	void *a,**hook; // The array and access hook
	unsigned m; // Number of members
	unsigned buffer,minbuffer,elementsize;
	float minpercentusage;
} LSet; // Should be 28 bytes

const LSet LSet_null={NULL,NULL, 0, 0,0,0, 60.0};

#define LSet(type) LSet_new(sizeof(type))
#define LSet_alloc(type,N) LSet_newalloc(sizeof(type),N,0)
#define LSet_allocspace(type,N) LSet_newalloc(sizeof(type),N,1)

int LSet_initalloc=1;

LSet LSet_new(const unsigned s)
{
	LSet ret;
	ret.m=0;
	ret.buffer=ret.minbuffer=LSet_initalloc;
	ret.elementsize=s;
	ret.a=mallocex(ret.buffer*ret.elementsize,"LSet buffer");
	ret.minpercentusage=60;
	ret.hook=NULL;
	return ret;
}

LSet LSet_newalloc(const unsigned s,const unsigned m,const unsigned space=0)
{ // Override LSet_initalloc for just this one set; if space is set, elements are already "there" to be overwritten
	LSet ret;
	ret.m=space?m:0;
	ret.buffer=ret.minbuffer=m;
	ret.elementsize=s;
	ret.a=mallocex(ret.buffer*ret.elementsize,"LSet buffer");
	ret.minpercentusage=60;
	ret.hook=NULL;
	return ret;
}

LSet LSet_copy(const LSet s)
{
	LSet ret;
	ret=s;
	ret.a=(void *)mallocex(ret.buffer*ret.elementsize,"LSet buffer from duplication");
	memcpy(ret.a,s.a,ret.buffer*ret.elementsize);
	ret.hook=NULL;
	return ret;
}

void LSet_fullwrite(LSet *s,FILE *out)
{ // "Full" save, includes memory options and allows large records
	fwrite(&s->m,sizeof(unsigned),1,out);
	fwrite(&s->elementsize,sizeof(unsigned),1,out);
	fwrite(&s->minbuffer,sizeof(unsigned),1,out);
	fwrite(&s->minpercentusage,sizeof(float),1,out);
	fwrite(s->a,s->elementsize,s->m,out);
} // Header = 16 bytes

void LSet_write(LSet *s,FILE *out)
{ // Condensed format for sets with records shorter than 256 bytes
	fwrite(&s->m,sizeof(unsigned),1,out);
	unsigned char sz=s->elementsize;
	fputc(sz,out);
	fwrite(s->a,s->elementsize,s->m,out);
} // Saves 11 fewer bytes per LSet (header = 5 bytes)

void LSet_save(LSet *s,const char *filename)
{
	FILE *out=fopen(filename,"wb");
	LSet_write(s,out);
	fclose(out);
}

LSet LSet_fullread(FILE *in)
{
	LSet ret;
	fread(&ret.m,sizeof(unsigned),1,in);
	fread(&ret.elementsize,sizeof(unsigned),1,in);
	fread(&ret.minbuffer,sizeof(unsigned),1,in);
	fread(&ret.minpercentusage,sizeof(float),1,in);
	ret.buffer=Maxu(ret.minbuffer,(int)(ret.m/(0.5+0.005*ret.minpercentusage)));
	ret.a=malloc(ret.buffer*ret.elementsize);
	fread(ret.a,ret.elementsize,ret.m,in);
	ret.hook=NULL;
	return ret;
}

LSet LSet_read(FILE *in)
{
	LSet ret;
	fread(&ret.m,sizeof(unsigned),1,in);
	ret.elementsize=fgetc(in);
	ret.minbuffer=LSet_initalloc;
	ret.minpercentusage=60;
	ret.buffer=Maxu(ret.minbuffer,(int)(ret.m/(0.5+0.005*ret.minpercentusage)));
	ret.a=malloc(ret.buffer*ret.elementsize);
	fread(ret.a,ret.elementsize,ret.m,in);
	ret.hook=NULL;
	return ret;
}

LSet LSet_load(const char *filename)
{
	FILE *in=fopen(filename,"rb");
	LSet ret=LSet_read(in);
	fclose(in);
	return ret;
}

void **LSet_hookup(LSet *s,void **h)
{ // To unhook, set s.hook=NULL or call this with h=NULL
	void **oh=s->hook;
	s->hook=h;
	if (h) *(s->hook)=s->a;
	return oh; // This returns the old hook: later, call LSet_hookup(s,oldhook) to return to previous state
}

unsigned LSet_add(LSet *s,const void *x)
{ // Equivalent to memcpy(LSet_addspace(s),x,s->elementsize);
	unsigned m=s->m,sz=s->elementsize;
	if (m>=s->buffer)
	{
		void *na;
		s->buffer=(int)((m+1.0)/(0.5+0.005*s->minpercentusage));
		na=mallocex(s->buffer*sz,"LSet buffer");
		memcpy(na,s->a,m*sz); free(s->a); s->a=na;
		if (s->hook) *(s->hook)=na;
	}
	memcpy((unsigned char *)s->a+m*sz,x,sz); s->m++;
	return m;
}

void *LSet_addspace(LSet *s,const unsigned n=1)
{ // Add an uninitialised entry (or n entries) and return a pointer to it
	unsigned m=s->m,sz=s->elementsize;
	if (m+n>s->buffer)
	{
		void *na;
		s->buffer=(int)((float)(m+n)/(0.5+0.005*s->minpercentusage));
		na=mallocex(s->buffer*sz,"LSet buffer");
		memcpy(na,s->a,m*sz); free(s->a); s->a=na;
		if (s->hook) *(s->hook)=na;
	}
	s->m+=n;
	return (unsigned char *)s->a+m*sz;
}

void LSet_merge(LSet *s,const LSet t)
{ // s and t must have the same element size
	memcpy(LSet_addspace(s,t.m),t.a,t.m*t.elementsize);
}

void LSet_addshift(LSet *s,const void *x,const unsigned n)
{ // Inserts before entry n, keeps entries in order but it's slower because more has to be moved
	unsigned m=s->m,sz=s->elementsize;
	if (m>=s->buffer)
	{
		void *na;
		s->buffer=(int)((m+1.0)/(0.5+0.005*s->minpercentusage));
		na=mallocex(s->buffer*sz,"LSet buffer");
		memcpy(na,s->a,m*sz); free(s->a); s->a=na;
		if (s->hook) *(s->hook)=na;
	}
	for (unsigned i=m;i>n;i--) memcpy((unsigned char *)s->a+i*sz,(unsigned char *)s->a+(i-1)*sz,sz);
	memcpy((unsigned char *)s->a+n*sz,x,sz); s->m++;
}

void LSet_trim(LSet *s)
{ // Resizes LSet so there is no wasted space, use when finished adding things
	unsigned m=s->m,sz=s->elementsize;
	void *na=mallocex(Maxu(1,m*sz),"LSet buffer");
	s->buffer=m;
	if (m>0) memcpy(na,s->a,m*sz); free(s->a); s->a=na;
	if (s->hook) *(s->hook)=na;
}

int LSet_indexof(const LSet *s,const void *x)
{
	int n;
	for (n=s->m-1;n>=0;n--) if (!memcmp((unsigned char *)s->a+n*s->elementsize,x,s->elementsize)) return n;
	return -1;
}

#define LSet_contains(s,x) (LSet_indexof(s,x)+1)

typedef int (*sortfn)(const void *,const void *);

void LSet_qsort(LSet *s,const sortfn cmp)
{ // cmp compares pointers to elements of the LSet
	qsort(s->a,s->m,s->elementsize,cmp);
}

void LSet_remove(LSet *s,const unsigned n)
{
	unsigned m=s->m-1,sz=s->elementsize;
	if (n>m) return;
	if (n<m) memcpy((unsigned char *)s->a+n*sz,(unsigned char *)s->a+m*sz,sz); s->m--;
	if ((float)m/s->buffer<0.01*s->minpercentusage && s->buffer>s->minbuffer)
	{
		void *na;
		s->buffer=Maxu(s->minbuffer,(int)((float)m/(0.5+0.005*s->minpercentusage)));
		na=mallocex(s->buffer*sz,"LSet buffer");
		memcpy(na,s->a,m*sz); free(s->a); s->a=na;
		if (s->hook) *(s->hook)=na;
	}
}

void LSet_removeshift(LSet *s,unsigned n)
{ // Keeps entries in order but it's slower because more has to be moved
	unsigned m=s->m-1,sz=s->elementsize;
	if (n>m) return;
	else s->m--;
	for (;n<m;n++) memcpy((unsigned char *)s->a+n*sz,(unsigned char *)s->a+(n+1)*sz,sz);
	if ((float)m/s->buffer<0.01*s->minpercentusage && s->buffer>s->minbuffer)
	{
		void *na;
		s->buffer=Maxu(s->minbuffer,(int)((float)m/(0.5+0.005*s->minpercentusage)));
		na=mallocex(s->buffer*sz,"LSet buffer");
		memcpy(na,s->a,m*sz); free(s->a); s->a=na;
		if (s->hook) *(s->hook)=na;
	}
}

void LSet_delete(LSet *s,const void *x)
{
	int i=LSet_indexof(s,x);
	if (i>=0) LSet_remove(s,i);
}

void LSet_clear(LSet *s)
{
	free(s->a);
	s->m=0; s->buffer=s->minbuffer;
	s->a=mallocex(s->buffer*s->elementsize,"LSet buffer (just cleared)");
	if (s->hook) *(s->hook)=s->a;
}

void LSet_freeref(LSet *s)
{
	if (s->hook) *(s->hook)=NULL;
	free(s->a); free(s);
}

void LSet_free(LSet s)
{
	if (s.hook) *(s.hook)=NULL;
	free(s.a);
}

// A "neater" interface using macros - creates unforeseen variable names though
#define LSArray(type,name) LSet name##LSet=LSet_new(sizeof(type)); type *name; LSet_hookup(&name##LSet,&name);
#define LSArray_elements(name) name##LSet.m
#define LSArray_add(name,thing) LSet_add(&name##LSet,&thing)
#define LSArray_remove(name,index) LSet_remove(&name##LSet,index)
#define LSArray_free(name) LSet_free(name##LSet)

#define SBCLIB_LSET
#endif
