#define REAL double
#include <stdio.h>
#include <array_1d.c>
#include <lset.c>
#include <V2.c>
#include <rnd/sq.c>
#include <muon1/constants.c>
#include <csv.c>
#include <rnd/striseg.c>

void magnet_error_console(const char *s) {printf("Error: %s\n",s); exit(1);}

void (*magnet_error)(const char *s)=magnet_error_console;

V2 unitwirefield(const V2 r)
{ // r = x_sample-x_wire; output is for 1 Amp
	return (2e-7/(r.x*r.x+r.y*r.y))*V2_new(-r.y,r.x);
}

V2 unitsheetfield(const V2 r,const REAL a)
{ // r = x_sample-x0_sheet; sheet extends from (0,0) to (a,0), a>0; output is for 1 Amp/metre
	return 2e-7*V2_new(
		-atan(r.x/r.y)+atan((r.x-a)/r.y),
		0.5*(log(r.x*r.x+r.y*r.y)-log(sq(r.x-a)+r.y*r.y)));
}

V2 unitsheetfieldv(const V2 r,const REAL b)
{ // r = x_sample-x0_sheet; sheet extends from (0,0) to (0,b), b>0; output is for 1 Amp/metre
	return 2e-7*V2_new(
		0.5*(log(r.x*r.x+sq(r.y-b))-log(r.x*r.x+r.y*r.y)),
		atan(r.y/r.x)-atan((r.y-b)/r.x));
}

V2 blockfield(const V2 r,const V2 a,const V2 Br)
{ // Block extends from (0,0) to s; Br is remnant field vector
	return (-Br.x/mu0)*unitsheetfield(r,a.x)+
		(Br.x/mu0)*unitsheetfield(r-V2_new(0,a.y),a.x)+
		(Br.y/mu0)*unitsheetfieldv(r,a.y)+
		(-Br.y/mu0)*unitsheetfieldv(r-V2_new(a.x,0),a.y);
}

V2 unitsheetfieldvec(const V2 r,const V2 a)
{ // r = x_sample-x0_sheet; sheet extends from (0,0) to a; output is for 1 Amp/metre
	REAL l=V2_norm(a),k=1.0/l;
	V2 f=unitsheetfield(k*V2_new(r.x*a.x+r.y*a.y,-r.x*a.y+r.y*a.x),l);
	return k*V2_new(f.x*a.x-f.y*a.y,f.x*a.y+f.y*a.x);
}

typedef struct {V2 o,s,Br;} Block;
typedef struct {V2 Br; LSet vs;} Poly; // vs has clockwise winding
typedef struct {V2 c,B; REAL r,p;} IronWire; // p=proportion filled from 0 to 1
typedef struct {V2 c; REAL r,I;} RoundWire; // I is total current in Amps (out of the screen)

int Block_inside(const Block b,V2 p) {p-=b.o; return p.x>0 && p.x<b.s.x && p.y>0 && p.y<b.s.y;}

void Block_flipx(Block *pb)
{ // Does not change B_r
	pb->o.x=-pb->o.x-pb->s.x;
}

void Block_flipy(Block *pb)
{ // Does not change B_r
	pb->o.y=-pb->o.y-pb->s.y;
}

REAL Block_area(const Block b) {return b.s.x*b.s.y;}

REAL Blocks_area(const LSet bs)
{
	REAL ret=0; Block *b=(Block *)bs.a;
	for (int n=bs.m-1;n>=0;n--) ret+=Block_area(b[n]);
	return ret;
}

Poly Poly_new(void) {Poly ret; ret.vs=LSet(V2); return ret;}

Poly Poly_copy(const Poly p) {Poly ret; ret.Br=p.Br; ret.vs=LSet_copy(p.vs); return ret;}

void Poly_free(Poly p) {LSet_free(p.vs);}

void Poly_addpoint(Poly *pp,const REAL x,const REAL y)
	{V2 p=V2_new(x,y); LSet_add(&pp->vs,&p);}

void Polys_free(LSet s)
{
	for (int n=s.m-1;n>=0;n--) Poly_free(((Poly *)s.a)[n]);
	LSet_free(s);
}

int linecollide(const V2 a,const V2 b,const V2 c,const V2 d)
{
	REAL denom=(d.y-c.y)*(b.x-a.x)-(d.x-c.x)*(b.y-a.y);
	if (fabs(denom)<1e-12) return 0;
	REAL ua=((d.x-c.x)*(a.y-c.y)-(d.y-c.y)*(a.x-c.x))/denom;
	if (ua<=1e-12 || ua>=1-1e-12) return 0;
	REAL ub=((b.x-a.x)*(a.y-c.y)-(b.y-a.y)*(a.x-c.x))/denom;
	return ub>1e-12 && ub<1-1e-12;
}

int Poly_inside(const Poly p,const V2 r)
{
	int n,m=p.vs.m; V2 *v=(V2 *)p.vs.a,o=v[0];
	for (n=m-1;n>=0;n--)
	{
		if (v[n].x<o.x) o.x=v[n].x; 
		if (v[n].y<o.y) o.y=v[n].y;
	}
	o.x-=1e-3; o.y-=1e-3;
	int ret=0;
	for (n=m-1;n>=0;n--) ret^=linecollide(o,r,v[n],v[(n+1)%m]);
	return ret;
}

int Poly_collide(const Poly p,const Poly q)
{
	V2 *u=(V2 *)p.vs.a,*v=(V2 *)q.vs.a;
	for (int n=p.vs.m-1;n>=0;n--) for (int i=q.vs.m-1;i>=0;i--)
		if (linecollide(u[n],u[(n+1)%p.vs.m],v[i],v[(i+1)%q.vs.m])) return 1;
	return 0;
}

int Polys_collide(const LSet ps)
{
	int n,i,m=ps.m; Poly *p=(Poly *)ps.a;
	for (n=m-1;n>=0;n--) for (i=n-1;i>=0;i--)
		if (Poly_collide(p[n],p[i])) return 1;
	return 0;
}

V2 Poly_field(const Poly p,const V2 r)
{
	int n,m=p.vs.m; V2 ret=V2_zero; V2 *v=(V2 *)p.vs.a,e;
	for (n=m-1;n>=0;n--)
	{
		e=v[(n+1)%m]-v[n];
		ret+=((p.Br.x*e.x+p.Br.y*e.y)/(mu0*V2_norm(e)))*unitsheetfieldvec(r-v[n],e);
	}
	return ret;
}

V2 Poly_force(const Poly p,const Poly q,const int steps=100)
{ // Outputs force exerted on p by q in Newtons/(metre depth)
	int n,i,m=p.vs.m; V2 ret=V2_zero; V2 *v=(V2 *)p.vs.a,e,b;
	for (n=m-1;n>=0;n--)
	{
		e=v[(n+1)%m]-v[n];
		b=V2_zero;
		for (i=steps-1;i>=0;i--) b+=Poly_field(q,v[n]+((0.5+i)/steps)*e);
		b/=steps;
		ret+=((p.Br.x*e.x+p.Br.y*e.y)/mu0)*V2_rotateright(b,-1);
	}
	return ret;	
}

void Poly_move(Poly *pp,const V2 x)
{
	for (int n=pp->vs.m-1;n>=0;n--) ((V2 *)pp->vs.a)[n]+=x;	
}

void Poly_rotateright(Poly *pp,const int r)
{ // Does not rotate B_r
	V2 *v=(V2 *)pp->vs.a;
	for (int n=pp->vs.m-1;n>=0;n--) v[n]=V2_rotateright(v[n],r);
}

void Poly_rotate(Poly *pp,const REAL th)
{ // Also rotates B_r
	V2 *v=(V2 *)pp->vs.a;
	for (int n=pp->vs.m-1;n>=0;n--) V2_rotate(&v[n],th);
	V2_rotate(&pp->Br,th);
}

void Poly_flipy(Poly *pp)
{ // Does not change B_r
	int n,m=pp->vs.m; V2 *v=(V2 *)pp->vs.a,u;
	for (n=m-1;n>=0;n--) v[n].y*=-1;
	for (n=m-1;n>=m/2;n--) {u=v[n]; v[n]=v[m-1-n]; v[m-1-n]=u;}
}

void Poly_scalex(Poly *pp,const REAL k)
{ // k>0 to preserve winding
	V2 *v=(V2 *)pp->vs.a;
	for (int n=pp->vs.m-1;n>=0;n--) v[n].x*=k;
}

V2 Poly_centre(const Poly p)
{ // Just average of vertices
	V2 *v=(V2 *)p.vs.a,ret=V2_zero;
	for (int n=p.vs.m-1;n>=0;n--) ret+=v[n];
	return ret/(double)p.vs.m;
}

void Poly_avoidvgap(Poly *pp,const REAL y)
{
	int n,m=pp->vs.m; V2 *v=(V2 *)pp->vs.a,c=Poly_centre(*pp);
	for (n=m-1;n>=0;n--) if (v[n].x<0 && fabs(v[n].y)<y) v[n].y=(c.y>0?y:-y);
}

V2 Poly_min(const Poly p)
{
	V2 ret=V2_new(9e99,9e99),*v=(V2 *)p.vs.a;
	for (int n=p.vs.m-1;n>=0;n--) ret=V2_min(ret,v[n]);
	return ret;
}

V2 Poly_max(const Poly p)
{
	V2 ret=V2_new(-9e99,-9e99),*v=(V2 *)p.vs.a;
	for (int n=p.vs.m-1;n>=0;n--) ret=V2_max(ret,v[n]);
	return ret;
}

REAL Poly_area(const Poly p)
{
	REAL ret=0;
	int n,m=p.vs.m; V2 *v=(V2 *)p.vs.a,a,b;
	for (n=m-1;n>=0;n--)
	{
		a=v[n]; b=v[(n+1)%m];
		ret+=a.y*b.x-a.x*b.y;
	}
	return 0.5*ret;
}

REAL Polys_area(const LSet ps)
{
	REAL ret=0; Poly *p=(Poly *)ps.a;
	for (int n=ps.m-1;n>=0;n--) ret+=Poly_area(p[n]);
	return ret;
}

IronWire IronWire_new(const V2 c,const REAL r,const REAL p=1)
	{IronWire ret; ret.c=c; ret.B=V2_zero; ret.r=r; ret.p=p; return ret;}

V2 dipole_fieldfv(const double x,const double y,const double Bx,const double By,const double r)
{ // Bx*fieldfx+By*fieldfy, calculated a bit faster
    double rr=(x*x+y*y)/(r*r);
    if (rr>=1) return V2_new(Bx*(x*x-y*y)+By*2*x*y,Bx*2*x*y+By*(y*y-x*x))/(rr*rr*r*r);
	else return V2_new(Bx,By);
}

V2 IronWire_field(const IronWire w,const V2 x)
{
	if (w.r<=0 || w.p==0 || w.B==V2_zero) return V2_zero;
	V2 p=x-w.c;
	return w.p*dipole_fieldfv(p.x,p.y,w.B.x,w.B.y,w.r);
}

V2 RoundWire_field(const RoundWire w,const V2 x)
{
	if (w.I==0) return V2_zero;
	V2 p=x-w.c; REAL pp=p.x*p.x+p.y*p.y;
	if (pp>=w.r*w.r) return (2e-7*w.I/pp)*V2_new(-p.y,p.x);
	else return (2e-7*w.I/(w.r*w.r))*V2_new(-p.y,p.x);
}

// The Magnet structure

typedef struct {LSet blocks,polys,irons,wires;} Magnet;

Magnet Magnet_new(void)
{
	Magnet ret;
	ret.blocks=LSet(Block);
	ret.polys=LSet(Poly);
	ret.irons=LSet(IronWire);
	ret.wires=LSet(RoundWire);
	return ret;
}

void Magnet_free(Magnet mag)
{
	LSet_free(mag.blocks);
	Polys_free(mag.polys);
	LSet_free(mag.irons);
	LSet_free(mag.wires);	
}

void Magnet_save(const Magnet mag,const char *filename,
	void (*ironwireqdesc)(const REAL,char *,const REAL)=NULL)
{
	int n,i,m; FILE *out=fopen(filename,"wt");
	fprintf(out,"Block name, B_rx (T),B_ry, x1 (m),y1,x2,y2,...\n");
	Block *b=(Block *)mag.blocks.a,c;
	for (n=0;n<mag.blocks.m;n++)
	{
		c=b[n];
		fprintf(out,"Block %d, %.9lg,%.9lg, %.9lg,%.9lg,%.9lg,%.9lg,%.9lg,%.9lg,%.9lg,%.9lg\n",
			1+n, c.Br.x,c.Br.y, c.o.x,c.o.y,c.o.x,c.o.y+c.s.y,c.o.x+c.s.x,c.o.y+c.s.y,c.o.x+c.s.x,c.o.y);
	}
	Poly *p=(Poly *)mag.polys.a; V2 *v;
	for (n=0;n<mag.polys.m;n++)
	{
		fprintf(out,"Poly %d, %.9lg,%.9lg, ",1+n,p[n].Br.x,p[n].Br.y);
		v=(V2 *)p[n].vs.a; m=p[n].vs.m;
		for (i=0;i<m;i++) fprintf(out,"%.9lg,%.9lg%c",v[i].x,v[i].y,i+1==m?'\n':',');
	}
	if (mag.irons.m)
	{
		fprintf(out,"Block name, B_xhere (T),B_yhere, cx (m),cy,r, proportion, Area (mm^2),Wires required\n");
		IronWire *w=(IronWire *)mag.irons.a; REAL mm2; char str[150];
		for (n=0;n<mag.irons.m;n++)
		{
			mm2=(M_PI*sq(w[n].r)*w[n].p)/1e-6;
			if (ironwireqdesc) ironwireqdesc(mm2,str,0.06); else *str=0;
			fprintf(out,"Iron wire %d, %.9lg,%.9lg, %.9lg,%.9lg,%.9lg, %.9lg, %.9lg,%s\n",1+n,
				w[n].B.x,w[n].B.y, w[n].c.x,w[n].c.y,w[n].r, w[n].p, mm2,str);
		}
	}
	if (mag.wires.m)
	{
		fprintf(out,"Block name, cx (m),cy,r, I (A)\n");
		RoundWire *w=(RoundWire *)mag.wires.a;
		for (n=0;n<mag.wires.m;n++)
			fprintf(out,"Copper wire %d, %.9lg,%.9lg,%.9lg, %.9lg\n",1+n,
				w[n].c.x,w[n].c.y,w[n].r, w[n].I);
	}
	fclose(out);
}

Magnet Magnet_fromcsv(const csv f)
{ // In-RAM loading (in absence of proper "magnets" structure, use csv)
	int n,i,rm,m=csvm(f); Magnet ret=Magnet_new();
	for (n=0;n<m;n++)
	{
		if (striseg(f[n][0],"Block name")) continue; // Header row
		else if (striseg(f[n][0],"Block"))
		{
			Block b;
			b.Br.x=atof(f[n][1]);
			b.Br.y=atof(f[n][2]);
			b.o.x=atof(f[n][3]);
			b.o.y=atof(f[n][4]);
			b.s.x=atof(f[n][7])-b.o.x;
			b.s.y=atof(f[n][6])-b.o.y;
			LSet_add(&ret.blocks,&b);
		}
		else if (striseg(f[n][0],"Poly"))
		{
			Poly p=Poly_new();
			p.Br.x=atof(f[n][1]);
			p.Br.y=atof(f[n][2]);
			rm=csvrowm(f[n]);
			for (i=4;i<rm;i+=2) Poly_addpoint(&p,atof(f[n][i-1]),atof(f[n][i]));
			LSet_add(&ret.polys,&p);
		}
		else if (striseg(f[n][0],"Iron wire"))
		{
			IronWire w;
			w.B.x=atof(f[n][1]);
			w.B.y=atof(f[n][2]);
			w.c.x=atof(f[n][3]);
			w.c.y=atof(f[n][4]);
			w.r=atof(f[n][5]);
			w.p=atof(f[n][6]);
			LSet_add(&ret.irons,&w);
		}		
		else if (striseg(f[n][0],"Copper wire"))
		{
			RoundWire w;
			w.c.x=atof(f[n][1]);
			w.c.y=atof(f[n][2]);
			w.r=atof(f[n][3]);
			w.I=atof(f[n][4]);
			LSet_add(&ret.wires,&w);
		}
		else {char str[250]; sprintf(str,"Unrecognised block type in '%s'",f[n][0]); magnet_error(str);}
	}
	return ret;
}

Magnet Magnet_load(const char *filename)
{
	csv f=csvload(filename);
	if (!f) {char str[250]; sprintf(str,"Can't load file '%s' in magnets_load",filename); magnet_error(str); return Magnet_new();}
	Magnet ret=Magnet_fromcsv(f);
	csvfree(f);
	return ret;
}

/*
int coilcollision(const Block b,const int exclude)
{ // Set exclude=-1 to do all
	int n,rep=0; BLock *a=(Block *)blocks.a;
	for (n=blocks.m-1;n>=0;n--) if (n!=exclude && c.r2>a[n].r1 && c.r1<a[n].r2) // R overlap
	{
		if (repeats>1) rep=Maxi(1-repeats,Mini(repeats-1,(int)floor(0.5*(c.z1+c.z2-(a[n].z1+a[n].z2))/repeatcell+0.5))); // floor here because (int) chops negative numbers
		if (c.z2>a[n].z1+rep*repeatcell && c.z1<a[n].z2+rep*repeatcell) return 1;
	}
	return 0;
}*/

V2 Magnet_field(const Magnet mag,const V2 p,const int useirons=-1)
{ // Returns B vector
	int n; V2 ret=V2_zero;
	//if (Polys_collide(&polys)) return ret;
	Block *b=(Block *)mag.blocks.a; 
	for (n=mag.blocks.m-1;n>=0;n--) ret+=
		blockfield(p-V2_new(b[n].o.x,b[n].o.y),b[n].s,b[n].Br);
	Poly *q=(Poly *)mag.polys.a;
	for (n=mag.polys.m-1;n>=0;n--) ret+=Poly_field(q[n],p);
	if (useirons)
	{
		IronWire *w=(IronWire *)mag.irons.a;
		for (n=mag.irons.m-1;n>=0;n--) if (n+1!=useirons) ret+=IronWire_field(w[n],p);	
	}
	RoundWire *w=(RoundWire *)mag.wires.a;
	for (n=mag.wires.m-1;n>=0;n--) ret+=RoundWire_field(w[n],p);
	return ret;
	//return blockfield(p,0.02,0.02,0,1.1);
	/*if (KEY(VK_SHIFT)) return 13.0*unitwirefield(p);
	else if (KEY(VK_SPACE)) {REAL a=0.1*mx/xres; return 13.0*unitsheetfield(p,a)/a;}
	else {REAL b=0.1*my/yres; return 13.0*unitsheetfieldv(p,b)/b;}
	return V2_new(p.y,p.x)+ // Normal quad (all these are positive)
		V2_new(2.0*p.x*p.y,p.x*p.x-p.y*p.y)+ // Normal sext
		V2_new(3.0*p.x*p.x*p.y-p.y*p.y*p.y,p.x*p.x*p.x-3.0*p.x*p.y*p.y)+ // Normal oct
		V2_new(p.x,-p.y)*0.3+ // Skew quad
		V2_new(p.x*p.x-p.y*p.y,-2.0*p.x*p.y)*0.2+ // Skew sext
		V2_new(p.x*p.x*p.x-3.0*p.x*p.y*p.y,-3.0*p.x*p.x*p.y+p.y*p.y*p.y)*4; // Skew oct */
}

V2 Magnet_magnetisation(const Magnet mag,const V2 p)
{ // Returns M vector
	int n; V2 ret=V2_zero;
	Block *b=(Block *)mag.blocks.a; 
	for (n=mag.blocks.m-1;n>=0;n--) if (Block_inside(b[n],p)) ret+=b[n].Br;
	Poly *q=(Poly *)mag.polys.a;
	for (n=mag.polys.m-1;n>=0;n--) if (Poly_inside(q[n],p)) ret+=q[n].Br;
	return ret;
}

REAL Magnet_area(const Magnet mag)
{
	return Blocks_area(mag.blocks)+Polys_area(mag.polys);
}

REAL Magnet_maxr(const Magnet mag)
{
	int n,i; V2 *v; REAL ret=0;
	Poly *p=(Poly *)mag.polys.a;
	for (n=mag.polys.m-1;n>=0;n--)
	{
		v=(V2 *)p[n].vs.a;
		for (i=p[n].vs.m-1;i>=0;i--) ret=Max(ret,V2_norm(v[i]));
	}
	return ret;
}

void Magnet_IronWires_init(Magnet *pm,const int IronWires_feedback)
{ // Call after geometry is updated but before fields are evaluated
	int n; IronWire *w=(IronWire *)pm->irons.a;
	if (IronWires_feedback) for (int i=5;i>0;i--)
		for (n=pm->irons.m-1;n>=0;n--) w[n].B=Magnet_field(*pm,w[n].c,n+1);
	else for (n=pm->irons.m-1;n>=0;n--) w[n].B=Magnet_field(*pm,w[n].c,0);		
}

V2 V2_degs(const REAL deg) {return V2_direction((90.0-deg)*M_PI/180.0);}

void (*Magnet_rotatingcoil_applyerrorset)(REAL *pn,REAL *ps,const REAL r)=NULL;

void Magnet_rotatingcoil(const Magnet mag,const V2 c,const REAL r,REAL *pn,REAL *ps,const int m=360)
{ // Fills pn,ps array_1ds (must be same size), units are Teslas at radius r
	int n,i,pm=array_1d_m(pn); V2 p,B,d,pd; REAL ox;
	array_1d_zero(pn); array_1d_zero(ps);
	for (n=m-1;n>=0;n--)
	{
		d=V2_degs(360.0*n/m);
		p=c+r*d;
		B=Magnet_field(mag,p);
		pd=V2_new(0,1);
		for (i=0;i<pm;i++)
		{
			pn[i]+=B.y*pd.y-B.x*pd.x; // Agrees with Animesh's By>0 for x>0 for positive normal pole convention
			ps[i]+=B.x*pd.y+B.y*pd.x; // Agrees with Animesh's Bx>0 for x>0 for positive skew pole convention
			ox=pd.x;
			pd.x=pd.x*d.x-pd.y*d.y;
			pd.y=ox*d.y+pd.y*d.x;
		}
	}
	array_1d_scale(pn,1.0/m); array_1d_scale(ps,1.0/m);	
	if (Magnet_rotatingcoil_applyerrorset) Magnet_rotatingcoil_applyerrorset(pn,ps,r);
}

REAL *Magnet_rotatingcoil_interleaved(const Magnet mag,const V2 c,const REAL r,const int pm,const int m=360)
{
	REAL *pn=array_1d(pm),*ps=array_1d(pm),*ret=array_1d(pm*2);
	Magnet_rotatingcoil(mag,c,r,pn,ps);
	for (int n=pm-1;n>=0;n--) {ret[n*2]=pn[n]; ret[n*2+1]=ps[n];}
	array_1d_free(pn); array_1d_free(ps);
	return ret;
}
